<?php
/* --------------------------------------------------------------
   ImageProcessingGdService.php 2023-02-01
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2023 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\ImageProcessing\App;

use Gambio\Admin\Modules\ImageProcessing\App\Data\ImageProcessingConfigurationRepository;
use Gambio\Admin\Modules\ImageProcessing\App\Processor\GDImageProcessor;
use Gambio\Admin\Modules\ImageProcessing\Model\ImageProcessingSettings;
use Gambio\Admin\Modules\ImageProcessing\Service\ImageProcessingService;
use Gambio\Core\Application\ValueObjects\Path;

/**
 * Class ImageProcessingGdService
 *
 * @package Gambio\Admin\Modules\ImageProcessing\App
 */
class ImageProcessingGdService implements ImageProcessingService
{
    private GDImageProcessor                       $processor;
    private ImageProcessingConfigurationRepository $configurationRepository;
    private Path                                   $path;
    
    
    public function __construct(
        GDImageProcessor                       $processor,
        ImageProcessingConfigurationRepository $configurationRepository,
        Path                                   $path
    ) {
        $this->processor               = $processor;
        $this->configurationRepository = $configurationRepository;
        $this->path                    = $path;
    }
    
    
    /**
     * @inheritDoc
     */
    public function process(string $imageFile): void
    {
        $this->processPopup($imageFile);
        $this->processInfoImage($imageFile);
        $this->processThumbnail($imageFile);
        $this->processGalleryImage($imageFile);
    }
    
    
    /**
     * @inheritDoc
     */
    public function processPopup(string $imageFile): void
    {
        $configuration = $this->configurationRepository->getPopupConfiguration();
        $source        = $this->originalImagePath($imageFile);
        
        $this->processor->process($source, "{$configuration->path()}/$imageFile", $configuration);
    }
    
    
    /**
     * @inheritDoc
     */
    public function processInfoImage(string $imageFile): void
    {
        $configuration = $this->configurationRepository->getInfoImageConfiguration();
        $source        = $this->originalImagePath($imageFile);
        
        $this->processor->process($source, "{$configuration->path()}/$imageFile", $configuration);
    }
    
    
    /**
     * @inheritDoc
     */
    public function processThumbnail(string $imageFile): void
    {
        $configuration = $this->configurationRepository->getThumbnailConfiguration();
        $source        = $this->originalImagePath($imageFile);
        
        $this->processor->process($source, "{$configuration->path()}/$imageFile", $configuration);
    }
    
    
    /**
     * @inheritDoc
     */
    public function processGalleryImage(string $imageFile): void
    {
        $configuration = $this->configurationRepository->getGalleryConfiguration();
        $source        = $this->originalImagePath($imageFile);
        
        $this->processor->process($source, "{$configuration->path()}/$imageFile", $configuration);
    }
    
    
    /**
     * @inheritDoc
     */
    public function processImage(string $imageFile, ImageProcessingSettings $settings): void
    {
        $configuration = $this->configurationRepository->getCustomConfiguration($settings);
        $source        = "{$configuration->path()}/$imageFile";
        $destination   = "{$configuration->destination()}/$imageFile";
        
        $this->processor->processCustom($source, $destination, $configuration);
    }
    
    
    /**
     * Returns the path to the original product images directory.
     *
     * @param string $imageFile
     *
     * @return string
     */
    private function originalImagePath(string $imageFile): string
    {
        return "{$this->path->base()}/images/product_images/original_images/$imageFile";
    }
}